/********************************************
re_cmpl.c
copyright 2008-2014,2016, Thomas E. Dickey
copyright 1991-1994,2014, Michael D. Brennan

This is a source file for mawk, an implementation of
the AWK programming language.

Mawk is distributed without warranty under the terms of
the GNU General Public License, version 2, 1991.
********************************************/

/*
 * $MawkId: re_cmpl.c,v 1.29 2016/09/18 19:50:53 tom Exp $
 */

/*  re_cmpl.c  */

#include "mawk.h"
#include "memory.h"
#include "scan.h"
#include "regexp.h"
#include "repl.h"

typedef struct re_node {
    RE_DATA re;			/* keep this first, for re_destroy() */
    STRING *sval;
    struct re_node *link;
} RE_NODE;

/* a list of compiled regular expressions */
static RE_NODE *re_list;

static char efmt[] = "regular expression compile failed (%s)\n%s";

/* compile a STRING to a regular expression machine.
   Search a list of pre-compiled strings first
*/
PTR
re_compile(STRING * sval)
{
    register RE_NODE *p;
    RE_NODE *q;
    char *s;

    /* search list */
    s = sval->str;
    p = re_list;
    q = (RE_NODE *) 0;

    while (p) {
	if (sval->len == p->sval->len
	    && memcmp(s, p->sval->str, sval->len) == 0) {
	    /* found */
	    if (!q)		/* already at front */
		goto _return;
	    else {		/* delete from list for move to front */
		q->link = p->link;
		goto found;
	    }
	} else {
	    q = p;
	    p = p->link;
	}
    }

    /* not found */
    p = ZMALLOC(RE_NODE);
    p->sval = sval;

    sval->ref_cnt++;
    p->re.anchored = (*s == '^');
    p->re.is_empty = (sval->len == 0);
    if (!(p->re.compiled = REcompile(s, sval->len))) {
	ZFREE(p);
	sval->ref_cnt--;
	if (mawk_state == EXECUTION)
	    rt_error(efmt, REerror(), s);
	else {			/* compiling */
	    compile_error(efmt, REerror(), s);
	    return (PTR) 0;
	}
    }

  found:
    /* insert p at the front of the list */
    p->link = re_list;
    re_list = p;

  _return:

#ifdef DEBUG
    if (dump_RE)
	REmprint(p->re.compiled, stderr);
#endif
    return refRE_DATA(p->re);
}

/* this is only used by da() */

char *
re_uncompile(PTR m)
{
    register RE_NODE *p;

    for (p = re_list; p; p = p->link)
	if (p->re.compiled == cast_to_re(m))
	    return p->sval->str;
#ifdef DEBUG
    bozo("non compiled machine");
#else
    return NULL;
#endif
}

#ifdef NO_LEAKS
void
re_destroy(PTR m)
{
    RE_NODE *p = (RE_NODE *) m;
    RE_NODE *q;
    RE_NODE *r;

    if (p != 0) {
	for (q = re_list, r = 0; q != 0; r = q, q = q->link) {
	    if (q == p) {
		free_STRING(p->sval);
		REdestroy(p->re.compiled);
		if (r != 0)
		    r->link = q->link;
		else
		    re_list = q->link;
		free(q);
		break;
	    }
	}
    }
}
#endif

/*=================================================*/
/*  replacement	 operations   */

/* create a replacement CELL from a STRING *  */

/* Here the replacement string gets scanned for &
 * which calls for using the matched text in the replacement
 * So to get a literal & in the replacement requires a convention
 * which is \& is literal & not a matched text &.
 * Here are the Posix rules which this code supports:
           \&   -->   &
	   \\   -->   \
	   \c   -->   \c    
	   &    -->   matched text 
	   
*/

/* FIXME  -- this function doesn't handle embedded nulls
   split_buff[] and MAX_SPLIT are obsolete, but needed by this
   function.  Putting
   them here is temporary until the rewrite to handle nulls.
*/

#define MAX_SPLIT  256		/* handle up to 256 &'s as matched text */
static STRING *split_buff[MAX_SPLIT];

static CELL *
REPL_compile(STRING * sval)
{
    VCount count = 0;
    register char *p = sval->str;
    register char *q;
    char *xbuff;
    CELL *cp;

    q = xbuff = (char *) zmalloc(sval->len + 1);

    while (1) {
	switch (*p) {
	case 0:
	    *q = 0;
	    goto done;

	case '\\':
	    if (p[1] == '&' || p[1] == '\\') {
		*q++ = p[1];
		p += 2;
		continue;
	    } else
		break;

	case '&':
	    /* if empty we don't need to make a node */
	    if (q != xbuff) {
		*q = 0;
		split_buff[count++] = new_STRING(xbuff);
	    }
	    /* and a null node for the '&'  */
	    split_buff[count++] = (STRING *) 0;
	    /*  reset  */
	    p++;
	    q = xbuff;
	    continue;

	default:
	    break;
	}

	*q++ = *p++;
    }

  done:
    /* if we have one empty string it will get made now */
    if (q > xbuff || count == 0)
	split_buff[count++] = new_STRING(xbuff);

    /* This will never happen */
    if (count > MAX_SPLIT)
	overflow("replacement pieces", MAX_SPLIT);

    cp = ZMALLOC(CELL);
    if (count == 1 && split_buff[0]) {
	cp->type = C_REPL;
	cp->ptr = (PTR) split_buff[0];
	USED_SPLIT_BUFF(0);
    } else {
	STRING **sp = (STRING **)
	(cp->ptr = zmalloc(sizeof(STRING *) * count));
	VCount j = 0;

	while (j < count) {
	    *sp++ = split_buff[j++];
	    USED_SPLIT_BUFF(j - 1);
	}

	cp->type = C_REPLV;
	cp->vcnt = count;
    }
    zfree(xbuff, sval->len + 1);
    return cp;
}

/* free memory used by a replacement CELL  */

void
repl_destroy(CELL *cp)
{
    register STRING **p;
    VCount cnt;

    if (cp->type == C_REPL) {
	free_STRING(string(cp));
    } else if (cp->ptr != 0) {	/* an C_REPLV           */
	p = (STRING **) cp->ptr;
	for (cnt = cp->vcnt; cnt; cnt--) {
	    if (*p) {
		free_STRING(*p);
	    }
	    p++;
	}
	zfree(cp->ptr, cp->vcnt * sizeof(STRING *));
    }
}

/* copy a C_REPLV cell to another CELL */

CELL *
replv_cpy(CELL *target, CELL *source)
{
    STRING **t, **s;
    VCount cnt;

    target->type = C_REPLV;
    cnt = target->vcnt = source->vcnt;
    target->ptr = (PTR) zmalloc(cnt * sizeof(STRING *));

    t = (STRING **) target->ptr;
    s = (STRING **) source->ptr;
    while (cnt) {
	cnt--;
	if (*s)
	    (*s)->ref_cnt++;
	*t++ = *s++;
    }
    return target;
}

/* here's our old friend linked linear list with move to the front
   for compilation of replacement CELLs	 */

typedef struct repl_node {
    struct repl_node *link;
    STRING *sval;		/* the input */
    CELL *cp;			/* the output */
} REPL_NODE;

static REPL_NODE *repl_list;

/* search the list (with move to the front) for a compiled
   separator.
   return a ptr to a CELL (C_REPL or C_REPLV)
*/

CELL *
repl_compile(STRING * sval)
{
    register REPL_NODE *p;
    REPL_NODE *q;
    char *s;

    /* search the list */
    s = sval->str;
    p = repl_list;
    q = (REPL_NODE *) 0;
    while (p) {
	if (strcmp(s, p->sval->str) == 0)	/* found */
	{
	    if (!q)		/* already at front */
		return p->cp;
	    else {		/* delete from list for move to front */
		q->link = p->link;
		goto found;
	    }

	} else {
	    q = p;
	    p = p->link;
	}
    }

    /* not found */
    p = ZMALLOC(REPL_NODE);
    p->sval = sval;
    sval->ref_cnt++;
    p->cp = REPL_compile(sval);

  found:
/* insert p at the front of the list */
    p->link = repl_list;
    repl_list = p;
    return p->cp;
}

/* return the string for a CELL or type REPL or REPLV,
   this is only used by da()  */

char *
repl_uncompile(CELL *cp)
{
    register REPL_NODE *p = repl_list;

    if (cp->type == C_REPL) {
	while (p) {
	    if (p->cp->type == C_REPL && p->cp->ptr == cp->ptr)
		return p->sval->str;
	    else
		p = p->link;
	}
    } else {
	while (p) {
	    if (p->cp->type == C_REPLV &&
		memcmp(cp->ptr, p->cp->ptr, cp->vcnt * sizeof(STRING *))
		== 0)
		return p->sval->str;
	    else
		p = p->link;
	}
    }

#ifdef DEBUG
    bozo("unable to uncompile an repl");
#else
    return NULL;
#endif
}

/*
  convert a C_REPLV to	C_REPL
     replacing the &s with sval
*/

CELL *
replv_to_repl(CELL *cp, STRING * sval)
{
    register STRING **p;
    STRING **sblock = (STRING **) cp->ptr;
    unsigned cnt, vcnt = cp->vcnt;
    size_t len;
    char *target;

#ifdef DEBUG
    if (cp->type != C_REPLV)
	bozo("not replv");
#endif

    p = sblock;
    cnt = vcnt;
    len = 0;
    while (cnt--) {
	if (*p)
	    len += (*p++)->len;
	else {
	    *p++ = sval;
	    sval->ref_cnt++;
	    len += sval->len;
	}
    }
    cp->type = C_REPL;
    cp->ptr = (PTR) new_STRING0(len);

    p = sblock;
    cnt = vcnt;
    target = string(cp)->str;
    while (cnt--) {
	memcpy(target, (*p)->str, (*p)->len);
	target += (*p)->len;
	free_STRING(*p);
	p++;
    }

    zfree(sblock, vcnt * sizeof(STRING *));
    return cp;
}

#ifdef NO_LEAKS
typedef struct _all_ptrs {
    struct _all_ptrs *next;
    PTR m;
} ALL_PTRS;

static ALL_PTRS *all_ptrs;
/*
 * Some regular expressions are parsed, and the pointer stored in the byte-code
 * where we cannot distinguish it from other constants.  Keep a list here, to
 * free on exit for auditing.
 */
void
no_leaks_re_ptr(PTR m)
{
    ALL_PTRS *p = calloc(1, sizeof(ALL_PTRS));
    p->next = all_ptrs;
    p->m = m;
    all_ptrs = p;
}

void
re_leaks(void)
{
    while (all_ptrs != 0) {
	ALL_PTRS *next = all_ptrs->next;
	re_destroy(all_ptrs->m);
	free(all_ptrs);
	all_ptrs = next;
    }

    while (repl_list != 0) {
	REPL_NODE *p = repl_list->link;
	free_STRING(repl_list->sval);
	free_cell_data(repl_list->cp);
	ZFREE(repl_list);
	repl_list = p;
    }
}
#endif
