/************************************************************************/
/* 									*/
/* MainWin.cc:	Implementation of the MainWin class			*/
/*		This class creates the main window and handles all	*/
/*		events in it.						*/
/*		Roland Krause 1998					*/
/*									*/
/************************************************************************/

// --- Include ----------------------------------------------------------

#include <malloc.h>			// Traditional memory management

#include <X11/Intrinsic.h>		// Header of Xt
#include <X11/StringDefs.h>		// Ressource definitions

// Header of XawPlus

#include <X11/XawPlus/Paned.h>		// The panel widget
#include <X11/XawPlus/AsciiText.h>	// The widget for the file list
#include <X11/XawPlus/Label.h>		// The label widget
#include <X11/XawPlus/Command.h>	// The command widget for the buttons

// Class definition of this class

#include "FileViewer.h"
#include "MainWin.h"

// --- Implementation of work procedures --------------------------------

// **********************************************************************
//
// ViewNextFile(): Asyncronous read of the tar output for the
//		   archive content. This procedure will be installed
//		   from the ViewFile() callback if needed.
//
// Arguments:	   clientData - Contains pointer to the file selector
//			        object and the MainWin object
//
// **********************************************************************

Boolean ViewNextFile(XtPointer clientData)
{
  ObjectCallbackStruct	*call = (ObjectCallbackStruct *)clientData;
  char			*FileName, *FileData;

  FileName = call->mainWin->GetNextSelection();
  if (FileName != NULL)
  {
    call->mainWin->shellNumber++;
    FileData = call->fileDialogue->GetFileFromArchive(FileName);
    new FileViewer(call->mainWin->mainShell, call->mainWin->shellNumber,
		   FileName, FileData);
    delete(FileData);
    return(False);	// Make shure we come back here
  }
  return(True);		// All files done
}

// --- Implementation of actions and callbacks --------------------------

// Callbacks and actions are not part of this class ! They are only a part of
// this module. That is the reason why all callbacks works on objects.
// Since the callbacks are defined static, they are not visibile outside
// of this module.

// **********************************************************************
//
// EndOfProg():	   Callback to quit this program
//
// Arguments:	   clientData - Contains pointer to the file selector
//				object and the MainWin object
//
// **********************************************************************

static void EndOfProg(Widget w, XtPointer clientData, XtPointer callData)
{
  ObjectCallbackStruct *myObjects = (ObjectCallbackStruct *)clientData;

  // Since we don't reach the end of the main function, we have to
  // delete all created objects here before we exit. If we don't do
  // this, the destructors of these objects are not reached.

  delete (myObjects->fileDialogue);
  delete (myObjects->extrDialogue);
  delete (myObjects->mainWin);
  exit(0);
}

// **********************************************************************
//
// OpenDialogue(): Callback to open the file dialogue
//
// Arguments:	    clientData - Contains pointer to the file selector
//				 object and MainWin object
//
// **********************************************************************

static void OpenDialogue(Widget w, XtPointer clientData, XtPointer callData)
{
  int	xPosition, yPosition;

  ObjectCallbackStruct *call = (ObjectCallbackStruct *)clientData;

  call->mainWin->GetPosition(&xPosition, &yPosition);
  call->fileDialogue->PopUp(xPosition+80, yPosition+40);
}

// **********************************************************************
//
// ViewFile():	Callback for the view button
//
// Arguments:	clientData - Contains pointer to the file selector
//			     object and the MainWin object
//
// **********************************************************************

static void ViewFile(Widget w, XtPointer clientData, XtPointer callData)
{
  ObjectCallbackStruct	*call = (ObjectCallbackStruct *)clientData;
  char			*FileName, *FileData;

  FileName = call->mainWin->GetSelection();
  if (FileName != NULL)
  {
    call->mainWin->shellNumber++;
    FileData = call->fileDialogue->GetFileFromArchive(FileName);
    new FileViewer(call->mainWin->mainShell, call->mainWin->shellNumber,
		   FileName, FileData);
    delete(FileData);

    // Maybe ther are additional selections. For this case we start
    // a work procedure to handle the other file asynchronously

    (void)XtAppAddWorkProc(XtWidgetToApplicationContext(w), ViewNextFile, clientData);
  }
}

// **********************************************************************
//
// ExtractFile(): Callback for the extract button
//
// Arguments:	clientData - Contains pointer to the file selector
//			     object and the MainWin object
//
// **********************************************************************

static void ExtractFile(Widget w, XtPointer clientData, XtPointer callData)
{
  int	xPosition, yPosition;
  char  *FileList;

  ObjectCallbackStruct *call = (ObjectCallbackStruct *)clientData;

  FileList = call->mainWin->GetCompleteSelection();
  if (FileList != NULL)
  {
     call->mainWin->GetPosition(&xPosition, &yPosition);
     call->extrDialogue->PopUp(xPosition+80, yPosition+40,
				call->fileDialogue->CurrentPath,
				FileList, call->fileDialogue->tar);
     delete(FileList);
  }
}

// --- Implementation of public class methods ---------------------------

// **********************************************************************
//
// MainWin():	The constructor of this class
//		Creates the main window
//
// Arguments:	shell    - The root shell widget of this application
//		selector - Pointer to the file selector object
//		dialogue - Pointer to the extract dialogue object
//
// **********************************************************************

MainWin::MainWin(Widget shell, FileSelect *selector, ExtrDialog *dialogue)
{
   Widget basePanel, buttonPanel;

   static XawTextSelectType SelectArray[] = {XawselectLine, XawselectNull};

   ObjCData.mainWin = this;		// Call data for the OpenDialogue() callback
   ObjCData.fileDialogue = selector;
   ObjCData.extrDialogue = dialogue;

   StartSel = EndSel = CurrPosition = NULL;
   mainShell   = shell;
   shellNumber = 0;

// At first we create the base panel

   basePanel =  XtVaCreateManagedWidget("basePanel", panedWidgetClass, shell, NULL);

// Now we create the widgets on the base panel

   buttonPanel= XtVaCreateManagedWidget("buttonPanel", panedWidgetClass, basePanel,
		XtNshowGrip, FALSE, XtNorientation, XtorientHorizontal, NULL);

   tarList=	XtVaCreateManagedWidget("selectList", asciiTextWidgetClass, basePanel,
		XtNdisplayCaret, FALSE, XtNselectTypes, SelectArray,
		XtNeditType, XawtextRead, XtNstring, "", NULL);

   message=	XtVaCreateManagedWidget("messLabel", labelWidgetClass, basePanel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, XtNlabel, "", NULL);

// In the next step we create all buttons on the buttonPanel

   quitButton=	XtVaCreateManagedWidget("quit", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(quitButton, XtNcallback, EndOfProg, (XtPointer)&ObjCData);

   openButton=	XtVaCreateManagedWidget("open", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(openButton, XtNcallback, OpenDialogue, (XtPointer)&ObjCData);

   viewButton=	XtVaCreateManagedWidget("view", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(viewButton, XtNcallback, ViewFile, (XtPointer)&ObjCData);

   extract=	XtVaCreateManagedWidget("extract", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(extract, XtNcallback, ExtractFile, (XtPointer)&ObjCData);

   extrAll=	XtVaCreateManagedWidget("extrAll", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);

   newButton=	XtVaCreateManagedWidget("new", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);

   (void)	XtVaCreateManagedWidget("title", labelWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
}


// **********************************************************************
//
// RunIt():	Realize the main window and runs the application
//
// Arguments:	app   - The application context of this application
//		shell - The root shell widget of this application
//
// **********************************************************************

void MainWin::RunIt(XtAppContext app, Widget shell)
{
   XtRealizeWidget(shell);

   // The file dialogue needs an initialization when the widget tree
   // is realized

   ObjCData.fileDialogue->PostInit(shell, tarList, message);
   XtAppMainLoop(app);
}


// **********************************************************************
//
// GetPosition(): Determine the current position of this window
//
// Arguments:	  x  - result for the x position
//		  y  - result for the y position
//
// **********************************************************************

inline void MainWin::GetPosition(int *x, int *y)
{
   XtVaGetValues(mainShell, XtNx, x, XtNy, y, NULL);
}

// **********************************************************************
//
// GetSelection(): Get one of the selected file entries
//		   Returns NULL if nothing is selected
//
// Arguments:	   None
//
// **********************************************************************

char *MainWin::GetSelection(void)
{
   static char	   ExtractFileName[TAR_PATH_LEN];
   XawTextPosition StartPos, EndPos;
   int		   EntryLength;
   char		   *FileName, *FileList;

   XawTextGetSelectionPos(tarList, &StartPos, &EndPos);

   if (StartPos != EndPos)
   {
      // Selection found: Extract the last file name in the list

      XtVaGetValues(tarList, XtNstring, &FileList, NULL);
      StartSel = &FileList[StartPos];

      // Determine position of EndSel: Sometimes EndPos points to
      // the newline character of the line, sometimes to the first
      // character of the next line. Make shure, that we always point
      // to the newline character.

      EndSel = &FileList[EndPos];
      while (*EndSel != '\n') EndSel--;
      CurrPosition = EndSel; CurrPosition--;

      // Search for the start of the last text line in the selection

      while ((CurrPosition > StartSel) && (*CurrPosition != '\n')) CurrPosition--;
      if (*CurrPosition == '\n') CurrPosition++;
      EntryLength = (int)(EndSel - CurrPosition);

      // Now we copy this line into our text buffer

      (void)strncpy(ExtractFileName, CurrPosition, EntryLength);
      ExtractFileName[EntryLength] = '\0';

      if ((FileName = strrchr(ExtractFileName, ' ')) != NULL) FileName++;
      return(FileName);
   }
   return(NULL);
}

// **********************************************************************
//
// GetNextSelection():	Get next selected entry after a GetSelection()
//		   	Returns NULL if no more entries are selected
//
// Arguments:	   	None
//
// **********************************************************************

char *MainWin::GetNextSelection(void)
{
   static char	   ExtractFileName[TAR_PATH_LEN];
   char		   *FileName;
   int		   EntryLength;

   if ((StartSel != EndSel) && (CurrPosition > StartSel))
   {
      // At first, we have to skip the newline character

      CurrPosition--;
      EndSel = CurrPosition;
      CurrPosition--;

      // Search for the start of the previous text line in the selection

      while ((CurrPosition > StartSel) && (*CurrPosition != '\n')) CurrPosition--;
      if (*CurrPosition == '\n') CurrPosition++;
      EntryLength = (int)(EndSel - CurrPosition);

      // Now we copy this line into our text buffer

      (void)strncpy(ExtractFileName, CurrPosition, EntryLength);
      ExtractFileName[EntryLength] = '\0';

      if ((FileName = strrchr(ExtractFileName, ' ')) != NULL) FileName++;
      return(FileName);
   }
   else		// All selected file names are read
   {
      StartSel = EndSel = CurrPosition = NULL;
      return (NULL);
   }
}

// **********************************************************************
//
// GetCompleteSelection(): Get the complete list of the selected file
//		   	   entries. Returns NULL if nothing is selected
//			   or problems with the memory management occurs.
//
// Arguments:		   None
//
//			   The caller has to release the allocated
//			   memory via delete.
//
// **********************************************************************

char *MainWin::GetCompleteSelection(void)
{
   XawTextPosition StartPos, EndPos;
   char		   *ExtractFiles = NULL, *FileList,
		   *Start, *End, *p, *CopyPos;

   XawTextGetSelectionPos(tarList, &StartPos, &EndPos);

   if (StartPos != EndPos)
   {
      // Selection found: Extract all file names of the selection

      XtVaGetValues(tarList, XtNstring, &FileList, NULL);
      Start = &FileList[StartPos];

      // Determine position of EndSel: Sometimes EndPos points to
      // the newline character of the line, sometimes to the first
      // character of the next line. Make shure, that we always point
      // to the newline character.

      End = &FileList[EndPos];
      while (*End != '\n') End--;

      if ((ExtractFiles = (char *)malloc(End - Start)) != NULL)
      {
	// Loop over all lines in the selection until the last
	// line is reached

	CopyPos = ExtractFiles;
	while ((Start < End) && (Start = strchr(Start, '\n')))
	{
	   // Search the start of the file name from the
	   // end of the line

	   p = Start;
	   while (*p != ' ') p--;
	   p++;

	   // Copy the file name

	   (void)strncpy(CopyPos, p, (size_t)(Start - p));
	   CopyPos += Start - p;
	   *CopyPos++ = ' '; *CopyPos = '\0';

	   Start++;		// Skip to the next line
        }
      }
   }
   return(ExtractFiles);
}
